<?php
require_once 'funcoes.php';

ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
error_reporting(E_ALL);

ini_set('log_errors', 1);
ini_set('error_log', 'erro.log');

// Dados padrão do autorizador
$autorizador_padrao = [
    'numero' => '17422651000172',
    'nome' => 'CF RJ CONTABILIDADE LTDA',
    'tipo' => 'PJ',
    'papel' => 'contratante'
];

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Content-Type: application/json');
    http_response_code(405);
    echo json_encode(["erro" => "Método não permitido. Use POST."]);
    exit;
}

$input = json_decode(file_get_contents("php://input"), true);

if (json_last_error() !== JSON_ERROR_NONE) {
    header('Content-Type: application/json');
    http_response_code(400);
    echo json_encode([
        "erro" => "JSON inválido: " . json_last_error_msg(),
        "input_recebido" => file_get_contents("php://input")
    ]);
    exit;
}

// Validação dos campos obrigatórios
$campos_obrigatorios = [
    'certificado_base64' => 'string',
    'senha_certificado' => 'string',
    'assinante' => [
        'numero' => 'string',
        'nome' => 'string',
        'tipo' => 'string',
        'papel' => 'string'
    ]
];

foreach ($campos_obrigatorios as $campo => $tipo) {
    if (is_array($tipo)) {
        if (!isset($input[$campo]) || !is_array($input[$campo])) {
            header('Content-Type: application/json');
            http_response_code(400);
            echo json_encode([
                "erro" => "Campo obrigatório ausente ou inválido: $campo",
                "campo_esperado" => $tipo,
                "campo_recebido" => isset($input[$campo]) ? gettype($input[$campo]) : "não definido"
            ]);
            exit;
        }
        foreach ($tipo as $subcampo => $subtipo) {
            if (!isset($input[$campo][$subcampo]) || !is_string($input[$campo][$subcampo])) {
                header('Content-Type: application/json');
                http_response_code(400);
                echo json_encode([
                    "erro" => "Campo obrigatório ausente ou inválido: $campo.$subcampo",
                    "tipo_esperado" => $subtipo,
                    "tipo_recebido" => isset($input[$campo][$subcampo]) ? gettype($input[$campo][$subcampo]) : "não definido"
                ]);
                exit;
            }
        }
    } else {
        if (!isset($input[$campo]) || !is_string($input[$campo])) {
            header('Content-Type: application/json');
            http_response_code(400);
            echo json_encode([
                "erro" => "Campo obrigatório ausente ou inválido: $campo",
                "tipo_esperado" => $tipo,
                "tipo_recebido" => isset($input[$campo]) ? gettype($input[$campo]) : "não definido"
            ]);
            exit;
        }
    }
}

// Decodificar o certificado base64
$certificado_decodificado = base64_decode($input['certificado_base64']);
if ($certificado_decodificado === false) {
    header('Content-Type: application/json');
    http_response_code(400);
    echo json_encode(["erro" => "Certificado inválido em base64"]);
    exit;
}

// Salvar o certificado temporariamente
$temp_cert_path = tempnam(sys_get_temp_dir(), 'cert_');
file_put_contents($temp_cert_path, $certificado_decodificado);

try {
    // Carregar o certificado
    $certificados = carregarCertificado($temp_cert_path, $input['senha_certificado']);
    
    // Montar o termo de autorização usando o autorizador padrão
    $xmlNaoAssinado = montarTermoAutorizacao($autorizador_padrao, $input['assinante']);
    
    // Assinar o XML
    $xmlAssinado = assinar($xmlNaoAssinado, $certificados);
    
    // Remover o arquivo temporário
    unlink($temp_cert_path);
    
    // Retornar o resultado
    header('Content-Type: application/json');
    echo json_encode([
        "sucesso" => true,
        "xml_assinado" => $xmlAssinado,
        "xml_assinado_base64" => base64_encode($xmlAssinado),
        "xml_nao_assinado" => $xmlNaoAssinado->asXML(),
        "xml_nao_assinado_base64" => base64_encode($xmlNaoAssinado->asXML())
    ]);
    
} catch (Exception $e) {
    // Remover o arquivo temporário em caso de erro
    if (isset($temp_cert_path) && file_exists($temp_cert_path)) {
        unlink($temp_cert_path);
    }
    
    header('Content-Type: application/json');
    http_response_code(500);
    echo json_encode([
        "sucesso" => false,
        "erro" => $e->getMessage()
    ]);
}
?> 